# Fair E Web Host Billing Panel - Deployment Guide

## Quick Start Deployment

### Option 1: Automated Installation (Recommended)

```bash
cd fairpanel
sudo chmod +x install.sh
sudo ./install.sh
```

The script will:
- Check system requirements
- Install Composer dependencies
- Create database and user
- Generate security keys
- Set file permissions
- Configure web server
- Import database schema

### Option 2: Manual Installation

Follow the detailed steps in `INSTALLATION.md`

## Pre-Deployment Checklist

### 1. Server Requirements
- [ ] PHP 8.1+ installed
- [ ] MySQL 5.7+ or MariaDB 10.3+ installed
- [ ] Composer installed
- [ ] Apache/Nginx installed
- [ ] SSL certificate ready
- [ ] Domain DNS configured

### 2. Required PHP Extensions
```bash
php -m | grep -E 'pdo|pdo_mysql|mbstring|curl|openssl|json|xml'
```

All should be present.

### 3. File Upload
- [ ] All files uploaded to server
- [ ] Correct directory structure maintained
- [ ] .env.example file present

### 4. Composer Dependencies
```bash
cd fairpanel
composer install --no-dev --optimize-autoloader
```

### 5. Environment Configuration
```bash
cp .env.example .env
nano .env
```

**Critical Settings:**
- APP_KEY (generate new)
- ENCRYPTION_KEY (generate new)
- Database credentials
- Google OAuth credentials
- Email settings

**Generate Keys:**
```bash
# APP_KEY
php -r "echo 'base64:' . base64_encode(random_bytes(32)) . PHP_EOL;"

# ENCRYPTION_KEY
php -r "echo bin2hex(random_bytes(32)) . PHP_EOL;"
```

### 6. Database Setup
```bash
mysql -u root -p
```

```sql
CREATE DATABASE faireweb_billing26 CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'faireweb_billing_adm'@'localhost' IDENTIFIED BY 'SecurePassword123!';
GRANT ALL PRIVILEGES ON faireweb_billing26.* TO 'faireweb_billing_adm'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

```bash
mysql -u faireweb_billing_adm -p faireweb_billing26 < database/schema.sql
```

### 7. File Permissions
```bash
sudo chown -R www-data:www-data /var/www/billing.fairewebhost.ca/fairpanel
sudo find /var/www/billing.fairewebhost.ca/fairpanel -type d -exec chmod 755 {} \;
sudo find /var/www/billing.fairewebhost.ca/fairpanel -type f -exec chmod 644 {} \;
sudo chmod -R 777 /var/www/billing.fairewebhost.ca/fairpanel/storage
sudo chmod -R 777 /var/www/billing.fairewebhost.ca/fairpanel/public/uploads
```

### 8. Web Server Configuration

**Apache:**
```bash
sudo nano /etc/apache2/sites-available/billing.fairewebhost.ca.conf
sudo a2ensite billing.fairewebhost.ca
sudo a2enmod rewrite
sudo systemctl restart apache2
```

**Nginx:**
```bash
sudo nano /etc/nginx/sites-available/billing.fairewebhost.ca
sudo ln -s /etc/nginx/sites-available/billing.fairewebhost.ca /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

### 9. SSL Certificate
```bash
sudo certbot --apache -d billing.fairewebhost.ca
# OR
sudo certbot --nginx -d billing.fairewebhost.ca
```

### 10. Google Services Setup

**Google OAuth:**
1. Google Cloud Console → Create Project
2. Enable Google+ API
3. Create OAuth 2.0 credentials
4. Add redirect URI: `https://billing.fairewebhost.ca/auth/google/callback`
5. Copy credentials to .env

**Google Analytics:**
1. Create GA4 property
2. Get Measurement ID (G-XXXXXXXXXX)
3. Add to .env

**Google Tag Manager:**
1. Create container
2. Get Container ID (GTM-XXXXXXX)
3. Add to .env

**Microsoft Clarity:**
1. Create project
2. Get Project ID
3. Add to .env

## Post-Deployment Tasks

### 1. Initial Access
Visit: `https://billing.fairewebhost.ca`

**Default Admin:**
- Email: admin@fairewebhost.ca
- Password: Admin@123

### 2. Security Hardening

**Change Admin Password:**
1. Login with default credentials
2. Go to Account Settings
3. Change password immediately

**Update .env Security:**
```env
APP_ENV=production
APP_DEBUG=false
MAINTENANCE_MODE=false
```

**Verify File Permissions:**
```bash
ls -la storage/
ls -la public/uploads/
```

### 3. Test Core Functionality

**Authentication:**
- [ ] Login with admin account
- [ ] Logout
- [ ] Login with Google OAuth
- [ ] Register new account
- [ ] Password reset

**Email System:**
- [ ] Welcome email received
- [ ] Password reset email works
- [ ] Invoice email works

**Analytics:**
- [ ] Google Analytics tracking
- [ ] Microsoft Clarity recording
- [ ] UTM parameters captured

**Admin Panel:**
- [ ] Dashboard loads
- [ ] Customer management works
- [ ] Service management works
- [ ] Invoice creation works

**Client Portal:**
- [ ] Dashboard loads
- [ ] Services visible
- [ ] Invoices visible
- [ ] Tickets work

### 4. Configure Payment Gateways

**PayPal:**
```env
PAYPAL_MODE=live
PAYPAL_CLIENT_ID=your_live_client_id
PAYPAL_CLIENT_SECRET=your_live_secret
PAYPAL_WEBHOOK_ID=your_webhook_id
```

**Stripe:**
```env
STRIPE_PUBLIC_KEY=pk_live_xxxxxxxx
STRIPE_SECRET_KEY=sk_live_xxxxxxxx
STRIPE_WEBHOOK_SECRET=whsec_xxxxxxxx
```

### 5. Configure WHM/cPanel Integration

```env
WHM_HOST=server.yourdomain.com
WHM_USERNAME=root
WHM_API_TOKEN=your_api_token
WHM_PORT=2087
```

### 6. Configure Domain Registrar

```env
RESELLERCLUB_AUTH_USERID=your_reseller_id
RESELLERCLUB_API_KEY=your_api_key
RESELLERCLUB_TEST_MODE=false
```

## Monitoring & Maintenance

### 1. Setup Monitoring

**Server Monitoring:**
- CPU usage
- Memory usage
- Disk space
- Network traffic

**Application Monitoring:**
- Error logs
- Access logs
- Database performance
- API response times

**Uptime Monitoring:**
- Use services like UptimeRobot
- Monitor: https://billing.fairewebhost.ca
- Alert on downtime

### 2. Backup Strategy

**Daily Backups:**
```bash
# Add to crontab: crontab -e
0 2 * * * /usr/local/bin/backup-billing.sh
```

**Backup Script:**
```bash
#!/bin/bash
DATE=$(date +%Y%m%d)
BACKUP_DIR="/backups/billing"

# Database backup
mysqldump -u faireweb_billing_adm -p'password' faireweb_billing26 > $BACKUP_DIR/db_$DATE.sql

# Files backup
tar -czf $BACKUP_DIR/files_$DATE.tar.gz /var/www/billing.fairewebhost.ca/fairpanel

# Keep only last 30 days
find $BACKUP_DIR -name "*.sql" -mtime +30 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +30 -delete
```

### 3. Log Rotation

**Application Logs:**
```bash
# Add to crontab
0 0 * * * find /var/www/billing.fairewebhost.ca/fairpanel/storage/logs -name "*.log" -mtime +30 -delete
```

**Web Server Logs:**
```bash
# Configure logrotate
sudo nano /etc/logrotate.d/billing
```

### 4. Security Updates

**Regular Updates:**
```bash
# Update Composer dependencies
cd /var/www/billing.fairewebhost.ca/fairpanel
composer update

# Update system packages
sudo apt update && sudo apt upgrade
```

**Security Checklist:**
- [ ] Keep PHP updated
- [ ] Keep MySQL updated
- [ ] Keep Composer dependencies updated
- [ ] Monitor security advisories
- [ ] Review access logs regularly
- [ ] Update SSL certificates before expiry

### 5. Performance Optimization

**Enable OPcache:**
```bash
sudo nano /etc/php/8.1/apache2/php.ini
```

```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=10000
opcache.revalidate_freq=2
```

**Enable Compression:**
```apache
# Apache
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>
```

**Database Optimization:**
```sql
-- Run monthly
OPTIMIZE TABLE users, hosting_accounts, invoices, orders;
```

## Troubleshooting Deployment Issues

### Issue: 500 Internal Server Error

**Check:**
```bash
# Apache logs
sudo tail -f /var/log/apache2/billing_error.log

# Nginx logs
sudo tail -f /var/log/nginx/billing_error.log

# Application logs
tail -f storage/logs/*.log

# PHP errors
sudo tail -f /var/log/php8.1-fpm.log
```

**Common Causes:**
- Incorrect file permissions
- Missing .env file
- Database connection failed
- PHP syntax errors

### Issue: Database Connection Failed

**Verify:**
```bash
# Test connection
mysql -u faireweb_billing_adm -p faireweb_billing26

# Check .env
cat .env | grep DB_

# Check MySQL status
sudo systemctl status mysql
```

### Issue: Composer Install Fails

**Solutions:**
```bash
# Update Composer
composer self-update

# Clear cache
composer clear-cache

# Install with verbose output
composer install -vvv

# Check PHP version
php -v
```

### Issue: File Permission Errors

**Fix:**
```bash
# Reset all permissions
sudo chown -R www-data:www-data /var/www/billing.fairewebhost.ca/fairpanel
sudo find /var/www/billing.fairewebhost.ca/fairpanel -type d -exec chmod 755 {} \;
sudo find /var/www/billing.fairewebhost.ca/fairpanel -type f -exec chmod 644 {} \;
sudo chmod -R 777 /var/www/billing.fairewebhost.ca/fairpanel/storage
sudo chmod -R 777 /var/www/billing.fairewebhost.ca/fairpanel/public/uploads
```

### Issue: Google OAuth Not Working

**Check:**
1. Redirect URI matches exactly in Google Console
2. Client ID and Secret correct in .env
3. HTTPS enabled
4. No browser extensions blocking
5. Cookies enabled

### Issue: Email Not Sending

**Check:**
```bash
# Test SMTP connection
telnet smtp.gmail.com 587

# Check .env email settings
cat .env | grep MAIL_

# Check PHP mail function
php -r "mail('test@example.com', 'Test', 'Test message');"
```

## Production Checklist

Before going live:

- [ ] All tests passed
- [ ] Default admin password changed
- [ ] .env configured correctly
- [ ] APP_DEBUG=false
- [ ] SSL certificate installed
- [ ] Backups configured
- [ ] Monitoring setup
- [ ] Email system tested
- [ ] Payment gateways tested
- [ ] Google OAuth tested
- [ ] Analytics verified
- [ ] Documentation reviewed
- [ ] Support team trained
- [ ] Maintenance plan in place

## Support

**Technical Support:**
- Email: support@fairewebhost.ca
- Documentation: https://docs.fairewebhost.ca
- Emergency: 24/7 for critical issues

**Deployment Assistance:**
- Professional installation available
- Server configuration support
- Custom integration services

## Version Information

- **Version:** 1.0.0
- **Release Date:** January 2026
- **PHP Version:** 8.1+
- **Database:** MySQL 5.7+ / MariaDB 10.3+
- **Framework:** Custom MVC
- **Standards:** 2026 Production Standards

---

**Congratulations on deploying Fair E Web Host Billing Panel!**

For ongoing support and updates, visit: https://fairewebhost.ca