<?php

namespace App\Controllers\Admin;

use Core\Controller;
use App\Models\User;

class CustomerController extends Controller
{
    public function index(): void
    {
        $page = $this->request->get('page', 1);
        $search = $this->request->get('search', '');
        
        $query = $this->database->table('users')->where('role', 'client');
        
        if ($search) {
            $query->where('email', 'LIKE', "%{$search}%")
                  ->orWhere('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%");
        }
        
        $customers = $query->orderBy('created_at', 'DESC')->get();
        
        $this->view('admin.customers.index', [
            'customers' => $customers,
            'search' => $search,
        ]);
    }

    public function show(int $id): void
    {
        $customer = $this->database->table('users')
            ->where('id', $id)
            ->where('role', 'client')
            ->first();
        
        if (!$customer) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        // Get customer's services
        $services = $this->database->table('hosting_accounts')
            ->where('user_id', $id)
            ->get();
        
        // Get customer's invoices
        $invoices = $this->database->table('invoices')
            ->where('user_id', $id)
            ->orderBy('created_at', 'DESC')
            ->get();
        
        // Get customer's tickets
        $tickets = $this->database->table('tickets')
            ->where('user_id', $id)
            ->orderBy('created_at', 'DESC')
            ->get();
        
        // Get customer's activity
        $activities = $this->database->table('activity_logs')
            ->where('user_id', $id)
            ->orderBy('created_at', 'DESC')
            ->limit(20)
            ->get();
        
        $this->view('admin.customers.show', [
            'customer' => $customer,
            'services' => $services,
            'invoices' => $invoices,
            'tickets' => $tickets,
            'activities' => $activities,
        ]);
    }

    public function store(): void
    {
        $data = $this->validate([
            'first_name' => 'required|min:2|max:50',
            'last_name' => 'required|min:2|max:50',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:8',
            'phone' => 'required',
            'company' => 'max:100',
            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'zip' => 'required',
            'country' => 'required',
        ]);

        $data['password'] = hash_password($data['password']);
        $data['role'] = 'client';
        $data['status'] = 'active';
        $data['is_admin'] = 0;
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['updated_at'] = date('Y-m-d H:i:s');

        try {
            $userId = $this->database->table('users')->insert($data);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'customer_created',
                'description' => "Created customer: {$data['email']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Customer created successfully')
                ->redirect('/admin/customers/' . $userId);
        } catch (\Exception $e) {
            logger('Customer creation error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to create customer']])
                ->withInput()
                ->back();
        }
    }

    public function update(int $id): void
    {
        $customer = $this->database->table('users')
            ->where('id', $id)
            ->where('role', 'client')
            ->first();
        
        if (!$customer) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }

        $data = $this->validate([
            'first_name' => 'required|min:2|max:50',
            'last_name' => 'required|min:2|max:50',
            'phone' => 'required',
            'company' => 'max:100',
            'address' => 'required',
            'city' => 'required',
            'state' => 'required',
            'zip' => 'required',
            'country' => 'required',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $data['updated_at'] = date('Y-m-d H:i:s');

        try {
            $this->database->table('users')
                ->where('id', $id)
                ->update($data);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'customer_updated',
                'description' => "Updated customer: {$customer['email']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Customer updated successfully')
                ->redirect('/admin/customers/' . $id);
        } catch (\Exception $e) {
            logger('Customer update error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to update customer']])
                ->withInput()
                ->back();
        }
    }

    public function delete(int $id): void
    {
        $customer = $this->database->table('users')
            ->where('id', $id)
            ->where('role', 'client')
            ->first();
        
        if (!$customer) {
            $this->json(['error' => 'Customer not found'], 404);
            return;
        }

        try {
            $this->database->table('users')->where('id', $id)->delete();
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'customer_deleted',
                'description' => "Deleted customer: {$customer['email']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);

            $this->response
                ->withSuccess('Customer deleted successfully')
                ->redirect('/admin/customers');
        } catch (\Exception $e) {
            logger('Customer deletion error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to delete customer']])
                ->back();
        }
    }
}