<?php

namespace App\Controllers\Admin;

use Core\Controller;

class OrderController extends Controller
{
    public function index(): void
    {
        $status = $this->request->get('status', 'all');
        
        $query = $this->database->table('orders')
            ->select('orders.*', 'users.first_name', 'users.last_name', 'users.email')
            ->join('users', 'orders.user_id', '=', 'users.id');
        
        if ($status !== 'all') {
            $query->where('orders.status', $status);
        }
        
        $orders = $query->orderBy('orders.created_at', 'DESC')->get();
        
        $this->view('admin.orders.index', [
            'orders' => $orders,
            'status' => $status,
        ]);
    }

    public function show(int $id): void
    {
        $order = $this->database->table('orders')
            ->select('orders.*', 'users.first_name', 'users.last_name', 'users.email', 'users.phone')
            ->join('users', 'orders.user_id', '=', 'users.id')
            ->where('orders.id', $id)
            ->first();
        
        if (!$order) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        $this->view('admin.orders.show', [
            'order' => $order,
        ]);
    }

    public function approve(int $id): void
    {
        $order = $this->database->table('orders')->where('id', $id)->first();
        
        if (!$order) {
            $this->json(['error' => 'Order not found'], 404);
            return;
        }

        try {
            $this->database->beginTransaction();
            
            // Update order status
            $this->database->table('orders')
                ->where('id', $id)
                ->update([
                    'status' => 'completed',
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'order_approved',
                'description' => "Approved order #{$order['order_number']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Order approved successfully')
                ->redirect('/admin/orders/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Order approval error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to approve order']])
                ->back();
        }
    }

    public function reject(int $id): void
    {
        $order = $this->database->table('orders')->where('id', $id)->first();
        
        if (!$order) {
            $this->json(['error' => 'Order not found'], 404);
            return;
        }

        $reason = $this->request->post('reason', '');

        try {
            $this->database->beginTransaction();
            
            // Update order status
            $this->database->table('orders')
                ->where('id', $id)
                ->update([
                    'status' => 'cancelled',
                    'notes' => $reason,
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'order_rejected',
                'description' => "Rejected order #{$order['order_number']}: {$reason}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Order rejected successfully')
                ->redirect('/admin/orders/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Order rejection error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to reject order']])
                ->back();
        }
    }
}