<?php

namespace App\Controllers\Admin;

use Core\Controller;

class ServiceController extends Controller
{
    public function index(): void
    {
        $status = $this->request->get('status', 'all');
        
        $query = $this->database->table('hosting_accounts')
            ->select('hosting_accounts.*', 'users.first_name', 'users.last_name', 'users.email', 'servers.name as server_name')
            ->join('users', 'hosting_accounts.user_id', '=', 'users.id')
            ->leftJoin('servers', 'hosting_accounts.server_id', '=', 'servers.id');
        
        if ($status !== 'all') {
            $query->where('hosting_accounts.status', $status);
        }
        
        $services = $query->orderBy('hosting_accounts.created_at', 'DESC')->get();
        
        $this->view('admin.services.index', [
            'services' => $services,
            'status' => $status,
        ]);
    }

    public function show(int $id): void
    {
        $service = $this->database->table('hosting_accounts')
            ->select('hosting_accounts.*', 'users.first_name', 'users.last_name', 'users.email', 'servers.name as server_name', 'servers.hostname')
            ->join('users', 'hosting_accounts.user_id', '=', 'users.id')
            ->leftJoin('servers', 'hosting_accounts.server_id', '=', 'servers.id')
            ->where('hosting_accounts.id', $id)
            ->first();
        
        if (!$service) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        $this->view('admin.services.show', [
            'service' => $service,
        ]);
    }

    public function suspend(int $id): void
    {
        $service = $this->database->table('hosting_accounts')->where('id', $id)->first();
        
        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
            return;
        }

        $reason = $this->request->post('reason', '');

        try {
            $this->database->beginTransaction();
            
            // Update service status
            $this->database->table('hosting_accounts')
                ->where('id', $id)
                ->update([
                    'status' => 'suspended',
                    'notes' => $reason,
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'service_suspended',
                'description' => "Suspended service: {$service['domain']} - {$reason}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // TODO: Call WHM API to suspend account
            
            $this->database->commit();

            $this->response
                ->withSuccess('Service suspended successfully')
                ->redirect('/admin/services/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Service suspension error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to suspend service']])
                ->back();
        }
    }

    public function unsuspend(int $id): void
    {
        $service = $this->database->table('hosting_accounts')->where('id', $id)->first();
        
        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
            return;
        }

        try {
            $this->database->beginTransaction();
            
            // Update service status
            $this->database->table('hosting_accounts')
                ->where('id', $id)
                ->update([
                    'status' => 'active',
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'service_unsuspended',
                'description' => "Unsuspended service: {$service['domain']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // TODO: Call WHM API to unsuspend account
            
            $this->database->commit();

            $this->response
                ->withSuccess('Service unsuspended successfully')
                ->redirect('/admin/services/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Service unsuspension error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to unsuspend service']])
                ->back();
        }
    }

    public function terminate(int $id): void
    {
        $service = $this->database->table('hosting_accounts')->where('id', $id)->first();
        
        if (!$service) {
            $this->json(['error' => 'Service not found'], 404);
            return;
        }

        $reason = $this->request->post('reason', '');

        try {
            $this->database->beginTransaction();
            
            // Update service status
            $this->database->table('hosting_accounts')
                ->where('id', $id)
                ->update([
                    'status' => 'terminated',
                    'notes' => $reason,
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'service_terminated',
                'description' => "Terminated service: {$service['domain']} - {$reason}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // TODO: Call WHM API to terminate account
            
            $this->database->commit();

            $this->response
                ->withSuccess('Service terminated successfully')
                ->redirect('/admin/services');
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Service termination error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to terminate service']])
                ->back();
        }
    }
}