<?php

namespace App\Controllers\Admin;

use Core\Controller;

class SettingsController extends Controller
{
    public function index(): void
    {
        // Get all settings
        $settingsData = $this->database->table('settings')->get();
        
        // Convert to key-value array
        $settings = [];
        foreach ($settingsData as $setting) {
            $settings[$setting['key']] = $setting['value'];
        }
        
        $this->view('admin.settings.index', [
            'settings' => $settings,
        ]);
    }

    public function update(): void
    {
        $data = $this->request->all();
        
        try {
            $this->database->beginTransaction();
            
            foreach ($data as $key => $value) {
                if ($key === '_csrf_token') {
                    continue;
                }
                
                // Check if setting exists
                $existing = $this->database->table('settings')
                    ->where('key', $key)
                    ->first();
                
                if ($existing) {
                    // Update existing setting
                    $this->database->table('settings')
                        ->where('key', $key)
                        ->update([
                            'value' => $value,
                            'updated_at' => date('Y-m-d H:i:s'),
                        ]);
                } else {
                    // Create new setting
                    $this->database->table('settings')->insert([
                        'key' => $key,
                        'value' => $value,
                        'type' => 'string',
                        'updated_at' => date('Y-m-d H:i:s'),
                    ]);
                }
            }
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $this->session->get('user_id'),
                'action' => 'settings_updated',
                'description' => 'Updated system settings',
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Settings updated successfully')
                ->redirect('/admin/settings');
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Settings update error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to update settings']])
                ->back();
        }
    }
}