<?php

namespace App\Controllers\Client;

use Core\Controller;

class TicketController extends Controller
{
    public function index(): void
    {
        $user = $this->getAuthUser();
        
        $status = $this->request->get('status', 'all');
        
        $query = $this->database->table('tickets')
            ->where('user_id', $user['id']);
        
        if ($status !== 'all') {
            $query->where('status', $status);
        }
        
        $tickets = $query->orderBy('created_at', 'DESC')->get();
        
        $this->view('client.tickets.index', [
            'tickets' => $tickets,
            'status' => $status,
        ]);
    }

    public function show(int $id): void
    {
        $user = $this->getAuthUser();
        
        $ticket = $this->database->table('tickets')
            ->where('id', $id)
            ->where('user_id', $user['id'])
            ->first();
        
        if (!$ticket) {
            $this->response->setStatusCode(404);
            $this->view('errors.404');
            return;
        }
        
        // Get ticket replies
        $replies = $this->database->table('ticket_replies')
            ->select('ticket_replies.*', 'users.first_name', 'users.last_name')
            ->join('users', 'ticket_replies.user_id', '=', 'users.id')
            ->where('ticket_replies.ticket_id', $id)
            ->orderBy('ticket_replies.created_at', 'ASC')
            ->get();
        
        $this->view('client.tickets.show', [
            'ticket' => $ticket,
            'replies' => $replies,
        ]);
    }

    public function create(): void
    {
        $user = $this->getAuthUser();
        
        $data = $this->validate([
            'subject' => 'required|min:5|max:255',
            'department' => 'required',
            'priority' => 'required|in:low,medium,high,urgent',
            'message' => 'required|min:10',
        ]);

        try {
            $this->database->beginTransaction();
            
            // Generate ticket number
            $ticketNumber = 'TKT-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
            
            // Create ticket
            $ticketId = $this->database->table('tickets')->insert([
                'user_id' => $user['id'],
                'ticket_number' => $ticketNumber,
                'subject' => $data['subject'],
                'department' => $data['department'],
                'priority' => $data['priority'],
                'status' => 'open',
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
            ]);
            
            // Add initial message as reply
            $this->database->table('ticket_replies')->insert([
                'ticket_id' => $ticketId,
                'user_id' => $user['id'],
                'message' => $data['message'],
                'is_staff' => 0,
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $user['id'],
                'action' => 'ticket_created',
                'description' => "Created ticket #{$ticketNumber}: {$data['subject']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Support ticket created successfully')
                ->redirect('/client/tickets/' . $ticketId);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Ticket creation error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to create ticket']])
                ->withInput()
                ->back();
        }
    }

    public function reply(int $id): void
    {
        $user = $this->getAuthUser();
        
        $ticket = $this->database->table('tickets')
            ->where('id', $id)
            ->where('user_id', $user['id'])
            ->first();
        
        if (!$ticket) {
            $this->json(['error' => 'Ticket not found'], 404);
            return;
        }

        $data = $this->validate([
            'message' => 'required|min:10',
        ]);

        try {
            $this->database->beginTransaction();
            
            // Add reply
            $this->database->table('ticket_replies')->insert([
                'ticket_id' => $id,
                'user_id' => $user['id'],
                'message' => $data['message'],
                'is_staff' => 0,
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            // Update ticket status
            $this->database->table('tickets')
                ->where('id', $id)
                ->update([
                    'status' => 'waiting',
                    'updated_at' => date('Y-m-d H:i:s'),
                ]);
            
            // Log activity
            $this->database->table('activity_logs')->insert([
                'user_id' => $user['id'],
                'action' => 'ticket_replied',
                'description' => "Replied to ticket #{$ticket['ticket_number']}",
                'ip_address' => $this->request->getIp(),
                'user_agent' => $this->request->getUserAgent(),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
            
            $this->database->commit();

            $this->response
                ->withSuccess('Reply added successfully')
                ->redirect('/client/tickets/' . $id);
        } catch (\Exception $e) {
            $this->database->rollback();
            logger('Ticket reply error: ' . $e->getMessage(), 'error');
            
            $this->response
                ->withErrors(['general' => ['Failed to add reply']])
                ->back();
        }
    }
}