<?php

namespace App\Services\PayPal;

use GuzzleHttp\Client;

class PayPalService
{
    private string $mode;
    private string $clientId;
    private string $clientSecret;
    private Client $client;
    private string $baseUrl;

    public function __construct()
    {
        $this->mode = config('services.paypal.mode', 'sandbox');
        $this->clientId = config('services.paypal.client_id');
        $this->clientSecret = config('services.paypal.client_secret');
        
        $this->baseUrl = $this->mode === 'live'
            ? 'https://api-m.paypal.com'
            : 'https://api-m.sandbox.paypal.com';
        
        $this->client = new Client([
            'base_uri' => $this->baseUrl,
        ]);
    }

    private function getAccessToken(): ?string
    {
        try {
            $response = $this->client->post('/v1/oauth2/token', [
                'auth' => [$this->clientId, $this->clientSecret],
                'form_params' => [
                    'grant_type' => 'client_credentials',
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            return $result['access_token'] ?? null;
        } catch (\Exception $e) {
            logger('PayPal getAccessToken error: ' . $e->getMessage(), 'error');
            return null;
        }
    }

    public function createOrder(float $amount, string $currency = 'USD', array $metadata = []): array
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return [
                    'success' => false,
                    'error' => 'Failed to get access token',
                ];
            }
            
            $response = $this->client->post('/v2/checkout/orders', [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'intent' => 'CAPTURE',
                    'purchase_units' => [
                        [
                            'amount' => [
                                'currency_code' => $currency,
                                'value' => number_format($amount, 2, '.', ''),
                            ],
                            'custom_id' => $metadata['invoice_id'] ?? '',
                        ],
                    ],
                    'application_context' => [
                        'return_url' => url('/payment/paypal/success'),
                        'cancel_url' => url('/payment/paypal/cancel'),
                    ],
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return [
                'success' => true,
                'order_id' => $result['id'],
                'approval_url' => $this->getApprovalUrl($result['links']),
                'data' => $result,
            ];
        } catch (\Exception $e) {
            logger('PayPal createOrder error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function captureOrder(string $orderId): array
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return [
                    'success' => false,
                    'error' => 'Failed to get access token',
                ];
            }
            
            $response = $this->client->post("/v2/checkout/orders/{$orderId}/capture", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['status'] === 'COMPLETED') {
                return [
                    'success' => true,
                    'transaction_id' => $result['purchase_units'][0]['payments']['captures'][0]['id'] ?? '',
                    'data' => $result,
                ];
            }
            
            return [
                'success' => false,
                'error' => 'Payment not completed',
                'data' => $result,
            ];
        } catch (\Exception $e) {
            logger('PayPal captureOrder error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getOrderDetails(string $orderId): array
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return [
                    'success' => false,
                    'error' => 'Failed to get access token',
                ];
            }
            
            $response = $this->client->get("/v2/checkout/orders/{$orderId}", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return [
                'success' => true,
                'data' => $result,
            ];
        } catch (\Exception $e) {
            logger('PayPal getOrderDetails error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function refundCapture(string $captureId, float $amount, string $currency = 'USD'): array
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return [
                    'success' => false,
                    'error' => 'Failed to get access token',
                ];
            }
            
            $response = $this->client->post("/v2/payments/captures/{$captureId}/refund", [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'amount' => [
                        'currency_code' => $currency,
                        'value' => number_format($amount, 2, '.', ''),
                    ],
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            if ($result['status'] === 'COMPLETED') {
                return [
                    'success' => true,
                    'refund_id' => $result['id'],
                    'data' => $result,
                ];
            }
            
            return [
                'success' => false,
                'error' => 'Refund not completed',
                'data' => $result,
            ];
        } catch (\Exception $e) {
            logger('PayPal refundCapture error: ' . $e->getMessage(), 'error');
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function verifyWebhook(array $headers, string $body, string $webhookId): bool
    {
        try {
            $accessToken = $this->getAccessToken();
            
            if (!$accessToken) {
                return false;
            }
            
            $response = $this->client->post('/v1/notifications/verify-webhook-signature', [
                'headers' => [
                    'Authorization' => "Bearer {$accessToken}",
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'transmission_id' => $headers['PAYPAL-TRANSMISSION-ID'] ?? '',
                    'transmission_time' => $headers['PAYPAL-TRANSMISSION-TIME'] ?? '',
                    'cert_url' => $headers['PAYPAL-CERT-URL'] ?? '',
                    'auth_algo' => $headers['PAYPAL-AUTH-ALGO'] ?? '',
                    'transmission_sig' => $headers['PAYPAL-TRANSMISSION-SIG'] ?? '',
                    'webhook_id' => $webhookId,
                    'webhook_event' => json_decode($body, true),
                ],
            ]);
            
            $result = json_decode($response->getBody()->getContents(), true);
            
            return ($result['verification_status'] ?? '') === 'SUCCESS';
        } catch (\Exception $e) {
            logger('PayPal verifyWebhook error: ' . $e->getMessage(), 'error');
            return false;
        }
    }

    private function getApprovalUrl(array $links): ?string
    {
        foreach ($links as $link) {
            if ($link['rel'] === 'approve') {
                return $link['href'];
            }
        }
        return null;
    }
}