#!/bin/bash

# Fair E Web Host Billing Panel - Installation Script
# This script automates the installation process

set -e

echo "=========================================="
echo "Fair E Web Host Billing Panel Installer"
echo "=========================================="
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo "Please run as root (use sudo)"
    exit 1
fi

# Check PHP version
PHP_VERSION=$(php -r 'echo PHP_VERSION;' 2>/dev/null || echo "0")
if [ "$(printf '%s\n' "8.1" "$PHP_VERSION" | sort -V | head -n1)" != "8.1" ]; then
    echo "Error: PHP 8.1 or higher is required. Current version: $PHP_VERSION"
    exit 1
fi

echo "✓ PHP version check passed ($PHP_VERSION)"

# Check if Composer is installed
if ! command -v composer &> /dev/null; then
    echo "Error: Composer is not installed"
    echo "Install Composer from: https://getcomposer.org"
    exit 1
fi

echo "✓ Composer is installed"

# Check if MySQL is installed
if ! command -v mysql &> /dev/null; then
    echo "Error: MySQL is not installed"
    exit 1
fi

echo "✓ MySQL is installed"

# Get installation directory
INSTALL_DIR=$(pwd)
echo ""
echo "Installation directory: $INSTALL_DIR"
echo ""

# Install Composer dependencies
echo "Installing Composer dependencies..."
composer install --no-dev --optimize-autoloader

echo "✓ Dependencies installed"

# Create .env file if it doesn't exist
if [ ! -f .env ]; then
    echo ""
    echo "Creating .env file..."
    cp .env.example .env
    
    # Generate APP_KEY
    APP_KEY=$(php -r "echo 'base64:' . base64_encode(random_bytes(32));")
    sed -i "s|APP_KEY=.*|APP_KEY=$APP_KEY|g" .env
    
    # Generate ENCRYPTION_KEY
    ENCRYPTION_KEY=$(php -r "echo bin2hex(random_bytes(32));")
    sed -i "s|ENCRYPTION_KEY=.*|ENCRYPTION_KEY=$ENCRYPTION_KEY|g" .env
    
    echo "✓ .env file created with generated keys"
else
    echo "✓ .env file already exists"
fi

# Create required directories
echo ""
echo "Creating required directories..."
mkdir -p storage/logs
mkdir -p storage/cache
mkdir -p storage/sessions
mkdir -p public/uploads

echo "✓ Directories created"

# Set permissions
echo ""
echo "Setting permissions..."
chown -R www-data:www-data $INSTALL_DIR
find $INSTALL_DIR -type d -exec chmod 755 {} \;
find $INSTALL_DIR -type f -exec chmod 644 {} \;
chmod -R 777 storage
chmod -R 777 public/uploads

echo "✓ Permissions set"

# Database setup
echo ""
echo "=========================================="
echo "Database Configuration"
echo "=========================================="
echo ""
read -p "Enter MySQL root password: " -s MYSQL_ROOT_PASSWORD
echo ""

read -p "Database name [faireweb_billing26]: " DB_NAME
DB_NAME=${DB_NAME:-faireweb_billing26}

read -p "Database user [faireweb_billing_adm]: " DB_USER
DB_USER=${DB_USER:-faireweb_billing_adm}

read -p "Database password: " -s DB_PASSWORD
echo ""

# Create database and user
echo ""
echo "Creating database and user..."
mysql -u root -p"$MYSQL_ROOT_PASSWORD" <<EOF
CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER IF NOT EXISTS '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASSWORD';
GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';
FLUSH PRIVILEGES;
EOF

echo "✓ Database created"

# Update .env with database credentials
sed -i "s|DB_DATABASE=.*|DB_DATABASE=$DB_NAME|g" .env
sed -i "s|DB_USERNAME=.*|DB_USERNAME=$DB_USER|g" .env
sed -i "s|DB_PASSWORD=.*|DB_PASSWORD=$DB_PASSWORD|g" .env

# Import database schema
echo ""
echo "Importing database schema..."
mysql -u "$DB_USER" -p"$DB_PASSWORD" "$DB_NAME" < database/schema.sql

echo "✓ Database schema imported"

# Web server configuration
echo ""
echo "=========================================="
echo "Web Server Configuration"
echo "=========================================="
echo ""
echo "Select your web server:"
echo "1) Apache"
echo "2) Nginx"
echo "3) Skip (configure manually)"
read -p "Enter choice [1-3]: " WEB_SERVER

case $WEB_SERVER in
    1)
        echo ""
        read -p "Enter domain name (e.g., billing.fairewebhost.ca): " DOMAIN
        
        # Update .env with domain
        sed -i "s|APP_URL=.*|APP_URL=https://$DOMAIN|g" .env
        
        # Create Apache virtual host
        cat > /etc/apache2/sites-available/$DOMAIN.conf <<EOF
<VirtualHost *:80>
    ServerName $DOMAIN
    ServerAdmin admin@$DOMAIN
    DocumentRoot $INSTALL_DIR/public

    <Directory $INSTALL_DIR/public>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/${DOMAIN}_error.log
    CustomLog \${APACHE_LOG_DIR}/${DOMAIN}_access.log combined
</VirtualHost>
EOF
        
        # Enable site and modules
        a2ensite $DOMAIN
        a2enmod rewrite
        systemctl restart apache2
        
        echo "✓ Apache configured"
        echo ""
        echo "To enable SSL, run: certbot --apache -d $DOMAIN"
        ;;
    2)
        echo ""
        read -p "Enter domain name (e.g., billing.fairewebhost.ca): " DOMAIN
        
        # Update .env with domain
        sed -i "s|APP_URL=.*|APP_URL=https://$DOMAIN|g" .env
        
        # Create Nginx server block
        cat > /etc/nginx/sites-available/$DOMAIN <<EOF
server {
    listen 80;
    server_name $DOMAIN;
    root $INSTALL_DIR/public;
    index index.php index.html;

    access_log /var/log/nginx/${DOMAIN}_access.log;
    error_log /var/log/nginx/${DOMAIN}_error.log;

    location / {
        try_files \$uri \$uri/ /index.php?\$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
EOF
        
        # Enable site
        ln -sf /etc/nginx/sites-available/$DOMAIN /etc/nginx/sites-enabled/
        nginx -t
        systemctl restart nginx
        
        echo "✓ Nginx configured"
        echo ""
        echo "To enable SSL, run: certbot --nginx -d $DOMAIN"
        ;;
    3)
        echo "Skipping web server configuration"
        ;;
esac

# Final instructions
echo ""
echo "=========================================="
echo "Installation Complete!"
echo "=========================================="
echo ""
echo "Next steps:"
echo ""
echo "1. Configure your .env file with:"
echo "   - Google OAuth credentials"
echo "   - Google Analytics ID"
echo "   - Microsoft Clarity ID"
echo "   - Email settings"
echo "   - Payment gateway credentials"
echo ""
echo "2. If you haven't already, install SSL certificate:"
echo "   sudo certbot --apache -d yourdomain.com"
echo "   OR"
echo "   sudo certbot --nginx -d yourdomain.com"
echo ""
echo "3. Access your installation:"
echo "   https://yourdomain.com"
echo ""
echo "4. Default admin credentials:"
echo "   Email: admin@fairewebhost.ca"
echo "   Password: Admin@123"
echo ""
echo "   ⚠️  IMPORTANT: Change the default password immediately!"
echo ""
echo "5. Read the documentation:"
echo "   - README.md - Overview"
echo "   - INSTALLATION.md - Detailed installation guide"
echo "   - HELP.md - User documentation"
echo ""
echo "For support: support@fairewebhost.ca"
echo ""
echo "=========================================="